const jwt = require("jsonwebtoken");

const Link = require("../../models/Link.js");
const Banner = require("../../models/Banner.js");
const Description = require("../../models/Description.js");
const LinksAddress = require("../../models/LinksAddress.js");
const Question = require("../../models/Question.js");


const Menu = require("../../models/Menu.js");
const MenuPlan = require("../../models/MenuPlan.js");
const MenuCategory = require("../../models/MenuCategory.js");
const Workgroup = require("../../models/Workgroup.js");
const L_S = require("../../models/L_S");
const L_C = require("../../models/L_C");
const L_Separator = require("../../models/L_Separator.js");
const M_MC = require("../../models/M_MC.js");
const MenuItme = require("../../models/MenuItem.js");
const Province = require("../../models/Province.js");
const City = require("../../models/City.js");
const { Op } = require("sequelize");
const Socialmedia = require("../../models/Socialmedia.js");
const Separator = require("../../models/Separator.js");
const Contact = require("../../models/Contact.js");
const Block = require("../../models/Block.js");
const MenuNotification = require("../../models/MenuNotification.js");


exports.getSocialmedia = async (req, res, next) => {
    try {
        const items = await Socialmedia.findAll({ attributes: { exclude: ['createdAt', 'updatedAt'] } });
        if (items.length == 0) {
            const error = new Error("هیچ شبکه اجتماعی ای در پایگاه داده ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items });
    } catch (err) {
        next(err);
    }
};

exports.getSeparators = async (req, res, next) => {
    try {
        const items = await Separator.findAll({ attributes: { exclude: ['createdAt', 'updatedAt'] } });
        if (items.length == 0) {
            const error = new Error("هیچ جدا کننده ای در پایگاه داده ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items });
    } catch (err) {
        next(err);
    }
};

exports.getContacts = async (req, res, next) => {
    try {
        const items = await Contact.findAll({ attributes: { exclude: ['createdAt', 'updatedAt'] } });
        if (items.length == 0) {
            const error = new Error("هیچ راه ارتباطی ای در پایگاه داده ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};


// MenuPlan

exports.getMenuPlans = async (req, res, next) => {

    try {
        const items = await MenuPlan.findAll({
            order: [
                ['id', 'DESC']
            ],
            where: { publish: true },
            attributes: { exclude: ["publish", "createdAt", "updatedAt"] },
        });

        if (items.length == 0) {
            const error = new Error("هیچ پلن منو ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};

exports.getWorkgroups = async (req, res, next) => {

    try {
        const items = await Workgroup.findAll({
            attributes: { exclude: ["createdAt", "updatedAt"] },
        });

        if (items.length == 0) {
            const error = new Error("هیچ گروه کاری ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};

exports.getMenuCategories = async (req, res, next) => {

    try {
        const items = await MenuCategory.findAll({
            include: [{
                model: Workgroup,
                as: "workgroup",
                attributes: ["id", "title"]
            }],
        });

        if (items.length == 0) {
            const error = new Error("هیچ دسته بندی منو ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};


// MenuItem

exports.getLink = async (req, res, next) => {
    const address = req.params.address;
    const authHeader = req.get("Authorization");
    let user = false

    try {
        const link = await Link.findOne({
            where: { address },
            attributes: { exclude: ["createdAt", "updatedAt"] },
        },);

        if (!link) {
            const error = new Error("لینک با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            const linkBlocks = await Block.findAll({
                where: { linkId: link.id },
                attributes: { exclude: ["createdAt", "updatedAt"] },
                order: [['sort', 'ASC']]
            },);


            let blocks = JSON.parse(JSON.stringify(linkBlocks));

            for (let i = 0; i < blocks.length; i++) {
                const block = blocks[i];
                let items;
                switch (block.type) {
                    case "socialmedia":
                        items = await L_S.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']], include: { model: Socialmedia, as: "data" } })
                        break;
                    case "contact":
                        items = await L_C.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']], include: { model: Contact, as: "data" } })
                        break;
                    case "separator":
                        items = await L_Separator.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']], include: { model: Separator, as: "data" } })
                        break;
                    case "banner":
                        items = await Banner.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']] })
                        break;
                    case "description":
                        items = await Description.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']] })
                        break;
                    case "linksAddress":
                        items = await LinksAddress.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']] })
                        break;
                    case "question":
                        items = await Question.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']] })
                        break;
                    default:
                        console.log("test");
                }
                blocks[i] = { ...block, items }
            }


            if (authHeader) {
                const token = authHeader.split(" ")[1]; //Bearer Token => ['Bearer', token]
                const decodedToken = jwt.verify(token, process.env.JWT_SECRET);
                const expired = Date.now() >= decodedToken.exp * 1000
                if (!expired) {
                    console.log(link.dataValues.userId);

                    if (link.dataValues.userId == decodedToken.user.userId) {
                        user = true
                    }
                }
            }

            res.status(200).json({ data: link, blocks, user });
        }
    } catch (err) {
        next(err);
    }
};

exports.getMenu = async (req, res, next) => {
    const authHeader = req.get("Authorization");
    let user = false
    try {

        const menu = await Menu.findOne({
            where: { address: req.params.address, active: true },
            attributes: { exclude: ["active", "workgroupId", "createdAt", "updatedAt"] },
            include: [{
                model: Workgroup,
                as: "workgroup",
                attributes: { exclude: ["createdAt", "updatedAt"] },
            },
            {
                model: M_MC,
                as: "categories",
                where: { active: true },
                required: false,
                attributes: { exclude: ["createdAt", "updatedAt"], },
                include: [
                    {
                        model: MenuCategory,
                        as: "category",
                        attributes: { exclude: ["createdAt", "updatedAt"], },
                    },
                    {
                        model: MenuItme,
                        as: "items",
                        where: { active: true },
                        required: false,
                        attributes: { exclude: ["createdAt", "updatedAt"] },
                    },
                ]
            },
            {
                model: City,
                as: "city",
                include: {
                    model: Province,
                    as: "province",
                }
            },
            {
                model: MenuNotification,
                as: "notification",
                required: false,
                where: { active: true },
            },
            ],
            order: [
                ["categories", 'sort', 'ASC'],
                ["categories", "items", 'sort', 'ASC'],
                ["notification", "id", 'DESC'],
            ],
        });


        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        }


        const start = new Date();
        const diffTime = Math.abs(start - menu.dataValues.endDate);
        let diffDays = Math.floor(diffTime / (1000 * 60 * 60 * 24));
        if (Date.parse(start) > Date.parse(menu.dataValues.endDate)) {
            menu.active = false;
            menu.save();
            diffDays = 0;
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        }

        console.log(authHeader);
        
        if (authHeader) {
            const token = authHeader.split(" ")[1]; //Bearer Token => ['Bearer', token]
            const decodedToken = jwt.verify(token, process.env.JWT_SECRET);
            const expired = Date.now() >= decodedToken.exp * 1000
            if (!expired) {
                if (menu.dataValues.userId == decodedToken.user.userId) {
                    user = true
                }
            }
        }

        res.status(200).json({ ...menu.dataValues, userId: user, day: diffDays, user });
    } catch (err) {
        next(err);
    }
};

exports.getMenus = async (req, res, next) => {
    const page = +req.query.page || 1;
    const limit = +req.query.size || 16;
    const offset = (page - 1) * limit;

    const workgroup = req.query.workgroup || null;
    const category = req.query.category || null;
    const city = req.query.city || null;
    const search = req.query.search || "";

    let workgroupWhere = workgroup ? { where: { id: workgroup } } : null;
    let categoryWhere = category ? { where: { categoryId: category } } : null;
    let cityWhere = city ? { where: { id: city } } : null;

    try {
        const items = await Menu.findAll({
            limit,
            offset,
            where: {
                name: {
                    [Op.like]: `%${search}%`
                },
                active: true
            },
            attributes: { exclude: ["active", "workgroupId", "createdAt", "updatedAt"] },
            include: [{
                model: Workgroup,
                as: "workgroup",
                required: true,
                attributes: { exclude: ["createdAt", "updatedAt"] },
                ...workgroupWhere
            },
            {
                model: M_MC,
                as: "categories",
                required: true,
                attributes: { exclude: ["createdAt", "updatedAt"] },
                ...categoryWhere
            },
            {
                model: City,
                as: "city",
                include: {
                    model: Province,
                    as: "province",
                },
                ...cityWhere
            },
            ]
        });
        if (!items.length === 0) {
            const error = new Error("منو دیجیتالی یافت نشد");
            error.statusCode = 404;
            throw error;
        }

        res.status(200).json({ items });
    } catch (err) {
        next(err);
    }
};

exports.getCities = async (req, res, next) => {
    try {

        const items = await Province.findAll({
            include: [
                {
                    model: City,
                    as: "cities",

                },
            ]
        });
        if (!items.length === 0) {
            const error = new Error("منو دیجیتالی یافت نشد");
            error.statusCode = 404;
            throw error;
        }

        res.status(200).json({ items });
    } catch (err) {
        next(err);
    }
};


