const shortid = require("shortid");
const fs = require("fs");


const MenuPlan = require("../../models/MenuPlan");
const Workgroup = require("../../models/Workgroup");
const MenuCategory = require("../../models/MenuCategory");

const { schemaMenuPlan, schemaWorkgroup, schemaMenuCategory } = require("../../models/secure/menuValidation");
// const MenuOrders = require("../../models/MenuOrders");
const Menu = require("../../models/Menu");
const MenuOrders = require("../../models/MenuOrders");


// MenuPlan

exports.getMenuPlans = async (req, res, next) => {

    try {
        const items = await MenuPlan.findAll({
            order: [
                ['id', 'DESC']
            ]
        });

        if (items.length == 0) {
            const error = new Error("هیچ پلن منو ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};

exports.getMenuPlan = async (req, res, next) => {
    try {
        const item = await MenuPlan.findByPk(req.params.id);
        if (!item) {
            const error = new Error("پلن منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json(item);
    } catch (err) {
        next(err);
    }
};

exports.createMenuPlan = async (req, res, next) => {
    try {
        await schemaMenuPlan.validate(req.body, { abortEarly: false });
        await MenuPlan.create(req.body);
        res.status(201).json({ message: "پلن منو جدید با موفقیت ساخته شد" });
    } catch (err) {
        next(err);
    }
};

exports.editMenuPlan = async (req, res, next) => {
    const item = await MenuPlan.findByPk(req.params.id);
    try {
        if (!item) {
            const error = new Error("پلن منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            await schemaMenuPlan.validate(req.body, { abortEarly: false });
            const { title, day, price, discount, publish } = req.body;
            item.title = title;
            item.day = day;
            item.price = price;
            item.discount = discount;
            item.publish = publish;
            await item.save();
            res.status(200).json({ message: "پلن منو با موفقیت ویرایش شد" });
        }
    } catch (err) {
        next(err);
    }
};

exports.deleteMenuPlan = async (req, res, next) => {
    try {
        const item = await MenuPlan.findByPk(req.params.id);
        if (!item) {
            const error = new Error(" پلن منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            await MenuPlan.destroy({ where: { id: req.params.id } });
        }
        res.status(200).json({ message: "پلن منو با موفقیت پاک شد" });
    } catch (err) {
        next(err);
    }
};



// Workgroup

exports.getWorkgroups = async (req, res, next) => {

    try {
        const items = await Workgroup.findAll({
            order: [
                ['id', 'DESC']
            ]
        });

        if (items.length == 0) {
            const error = new Error("هیچ گروه کاری ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};

exports.getWorkgroup = async (req, res, next) => {
    try {
        const item = await Workgroup.findByPk(req.params.id);
        if (!item) {
            const error = new Error("گروه کاری با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json(item);
    } catch (err) {
        next(err);
    }
};

exports.createWorkgroup = async (req, res, next) => {
    try {
        await schemaWorkgroup.validate(req.body, { abortEarly: false });
        await Workgroup.create(req.body);
        res.status(201).json({ message: "گروه کاری جدید با موفقیت ساخته شد" });
    } catch (err) {
        next(err);
    }
};

exports.editWorkgroup = async (req, res, next) => {
    const item = await Workgroup.findByPk(req.params.id);
    try {
        if (!item) {
            const error = new Error("گروه کاری با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            await schemaWorkgroup.validate(req.body, { abortEarly: false });
            item.title = req.body.title;
            await item.save();
            res.status(200).json({ message: "گروه کاری با موفقیت ویرایش شد" });
        }
    } catch (err) {
        next(err);
    }
};

exports.deleteWorkgroup = async (req, res, next) => {
    try {
        const item = await Workgroup.findByPk(req.params.id);
        if (!item) {
            const error = new Error("گروه کاری با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            await Workgroup.destroy({ where: { id: req.params.id } });
        }
        res.status(200).json({ message: "گروه کاری با موفقیت پاک شد" });
    } catch (err) {
        next(err);
    }
};



// MenuCategory

exports.getMenuCategories = async (req, res, next) => {
    const page = +req.query.page || 1;
    const limit = +req.query.size || 8;
    const offset = (page - 1) * limit;


    try {
        const items = await MenuCategory.findAll({
            limit,
            offset,
            order: [
                ['id', 'DESC']
            ]
        });

        if (items.length == 0) {
            const error = new Error("هیچ دسته بندی ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }

        const count = await MenuCategory.count();
        const infoPage = {
            currentPage: page,
            nextPage: page + 1,
            previousPage: page - 1,
            hasNextPage: limit * page < count,
            hasPreviousPage: page > 1,
            pagesCount: Math.ceil(count / limit),
        }

        res.status(200).json({ items: items, count, infoPage });
    } catch (err) {
        next(err);
    }
};

exports.getMenuCategory = async (req, res, next) => {
    try {
        const item = await MenuCategory.findByPk(req.params.id);
        if (!item) {
            const error = new Error("دسته بندی با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json(item);
    } catch (err) {
        next(err);
    }
};

exports.createMenuCategory = async (req, res, next) => {
    try {

        const data = {
            title: req.body.title,
            svg: req.body.svg,
            workgroupId: req.body.workgroupId
        };

        await schemaMenuCategory.validate(data, { abortEarly: false });
        const workgroup = await Workgroup.findByPk(req.body.workgroupId);
        if (!workgroup) {
            const error = new Error(`گروه کاری با شناسه ${req.body.workgroupId} وجود ندارد`);
            error.statusCode = 404;
            throw error;
        }
        const newMenuCategory = await MenuCategory.create(data);

        res.status(201).json({ data: newMenuCategory, message: "دسته بندی جدید با موفقیت ساخته شد" });

    } catch (err) {
        next(err);
    }
};

exports.editMenuCategory = async (req, res, next) => {
    const item = await MenuCategory.findByPk(req.params.id);

    try {

        if (!item) {
            const error = new Error("دسته بندی با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            await schemaMenuCategory.validate(req.body);
            const workgroup = await Workgroup.findByPk(req.body.workgroupId);
            if (!workgroup) {
                const error = new Error(`گروه کاری با شناسه ${req.body.workgroupId} وجود ندارد`);
                error.statusCode = 404;
                throw error;
            }
            const { title, svg, workgroupId } = req.body;
            item.title = title;
            item.svg = svg;
            item.workgroupId = workgroupId;
            await item.save();

            res.status(200).json({ message: "دسته بندی منو با موفقیت ویرایش شد" });
        }
    } catch (err) {
        next(err);
    }
};

exports.deleteMenuCategory = async (req, res, next) => {
    try {
        const item = await MenuCategory.findByPk(req.params.id);
        if (!item) {
            const error = new Error("دسته بندی منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            await MenuCategory.destroy({ where: { id: req.params.id } });
        }
        res.status(200).json({ message: "دسته بندی منو با موفقیت پاک شد" });
    } catch (err) {
        next(err);
    }
};



// Menu

exports.getMenus = async (req, res, next) => {

    try {
        const items = await Menu.findAll();

        if (items.length == 0) {
            const error = new Error("هیچ منو ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};

exports.getMenuOrders = async (req, res, next) => {

    try {

        const items = await MenuOrders.findAll({
            attributes: { exclude: ["authority", "refId", "menuId", "planId", "updatedAt"] },
            include: [{
                model: Menu,
                as: "menu",
                attributes: ["id", "name", "address"],
            },
            {
                model: MenuPlan,
                as: "plan",
                attributes: { exclude: ["publish", "createdAt", "updatedAt"] },
            }
            ],
        });


        if (items.length == 0) {
            const error = new Error("هیچ منو ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};

exports.getConfirmMenus = async (req, res, next) => {

    try {
        const items = await Menu.findAll({
            attributes: { exclude: ["workgroupId", "userId", "updatedAt"] },
            where: { status: '1' },
        });

        if (items.length == 0) {
            const error = new Error("هیچ منو ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};

exports.confirmMenu = async (req, res, next) => {
    const confirm = req.body.confirm;

    try {
        const menu = await Menu.findOne({
            where: { id: req.params.id, status: '1' },
            include: [{
                model: MenuOrders,
                as: "orders",
                where: { isPay: true },
                include: { model: MenuPlan, as: "plan" },
                required: true
            }],
        });
        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        }
        const plan = menu.dataValues.orders[0].plan;
        if (confirm) {
            const start = new Date();
            start.setUTCHours(0, 0, 0, 0);
            const end = new Date(new Date().setDate(start.getDate() + plan.day));
            end.setUTCHours(23, 59, 59, 999);
            menu.startDate = start;
            menu.endDate = end;
            menu.active = true;
            menu.status = '2';
        } else {
            menu.status = '3';
        }

        menu.save();

        res.status(200).json({ message: "وضعیت اشتراک منو با موفقیت ویرایش شد" });
    } catch (err) {
        next(err);
    }
};